function p = parseargs( varargin )
%PARSEARGS  Parse and join key-val and structure arguments.
%
%   p = parseargs( [ key, value ], ... [ struct_pars ], [ cell_pars ], ... )
%
%   Builds a structure of a parameters. The parameters are specified by key-value
%   pairs, using a structure or cell matrix. Keys must be valid structure
%   fieldnames.
%
%   1) If an argument is a string, then it and the following one is treated as
%   key-value pair, forming a field with a value in the structure.
%
%   2) If an argument is a structure, than its fields with values are added.
%
%   3) If an argument is cell, the cell contents is parsed recursively (for
%   key-value pairs, structures and cell matrix again).
%
%   In case of repeating keys, the last value is preferred. So when the default
%   values must be put prior to the variable data.
%
%   Example:
%
%     p = parseargs( varargin )
%         .. all arguments of a function are parsed
%
%     p = parseargs( 'x', 1, 'y', 2, varargin )
%         .. in varargin does not specify the value for 'x' and/or 'y', there are
%            defaults to use (so p.x and p.y are always defined).
%
%     p1 = parseargs( 'key1', 1, 'key2', 'hey', 'key3', 'huh', 'key4', 'x' )
%
%     p2 = parseargs( 'key1', 1, ...
%                     struct( 'key2', 'hey', 'key3', 'huh' ), ...
%                     { 'key4', 'x' } )
%
%     p3 = parseargs( { 'key1', 1, { 'key2', 'hey' }, ...
%                       struct( 'key3', 'huh', 'key4', 'x' ) } )
%          .. these examples produce the same output in p1, p2 and p3

% (c) 2002-03-22 Martin Matousek, Czech Technical University in Prague
% Last change: $Date: 2017-06-12 17:31:54 +0200$
%              $Revision: c720398$

p = struct;

i = 1;
l = length( varargin ); % cache the length, update only when changed

while( i <= l )
  if( ischar( varargin{i} ) )           % a key-value pair
    if( i == l )
      tx = [ 'Missing value to key ''' varargin{i} '''' ];
      throwAsCaller( MException( '', tx ) );
    end
    p.( varargin{i} ) = varargin{i+1};
    i = i + 2;                          % move i over this key followed by value

  elseif( isstruct( varargin{i} ) )     % structure
    for f = fieldnames( varargin{i} )';
      p.(f{1}) = varargin{i}.(f{1});
    end
    i = i + 1;                          % move i over this structure

  elseif( iscell( varargin{i} ) )       % cell - interpolate (recursively)
    varargin = [ varargin(1:i-1) varargin{i} varargin(i+1:end) ];
    l = length( varargin );             % update length
    % do not move i - process varargin{i} again in the next step

  else
    throwAsCaller(  MException( '', 'Wrong argument' ) );
  end
end

